import React, { useState } from 'react';
import { NavLink } from 'react-router-dom';
import {
  Users, Briefcase, CheckSquare, LayoutGrid, Megaphone,
  Globe, Settings,
  Target, Bot, Package, Calendar, MessageSquare,
  Pin, PinOff, LayoutDashboard
} from 'lucide-react';

interface SidebarItemProps {
  icon: React.ReactNode;
  label: string;
  to: string;
  badge?: number;
  showFull: boolean;
}

const SidebarItem: React.FC<SidebarItemProps> = ({ icon, label, to, badge, showFull }) => {
  return (
    <NavLink
      to={to}
      className={({ isActive }) => `flex items-center py-3 cursor-pointer group transition-all duration-200 relative decoration-transparent ${isActive ? 'bg-white/20 text-white' : 'text-white/90 hover:bg-white/10 hover:text-white'}`}
    >
      {({ isActive }) => (
        <>
          {isActive && <div className="absolute left-0 top-0 bottom-0 w-1 bg-[#2fc6f6] shadow-[0_0_10px_#2fc6f6]"></div>}

          {/* Icon Wrapper - always centered in 16px (w-16) */}
          <div className="w-16 shrink-0 flex items-center justify-center">
            <span className={`${isActive ? 'text-[#2fc6f6]' : 'text-white/70 group-hover:text-[#2fc6f6]'} transition-colors shrink-0`}>
              {icon}
            </span>
          </div>

          {/* Label - slides out */}
          <div className={`flex-1 flex items-center justify-between pr-5 overflow-hidden transition-all duration-300 ${showFull ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4 pointer-events-none'}`}>
            <span className="text-[13px] font-semibold tracking-tight whitespace-nowrap">
              {label}
            </span>

            {badge !== undefined && (
              <span className={`text-white text-[9px] font-black px-1.5 py-1 rounded-full min-w-[18px] text-center shadow-lg bg-[#ff5752]`}>
                {badge}
              </span>
            )}
          </div>

          {/* Small badge dot for collapsed mode */}
          {badge !== undefined && !showFull && (
            <div className={`absolute top-2 left-10 w-2 h-2 rounded-full border border-white/20 ${badge > 5 ? 'bg-[#ff5752]' : 'bg-[#2fc6f6]'}`}></div>
          )}
        </>
      )}
    </NavLink>
  );
};

interface SidebarProps {
  isLocked: boolean;
  setIsLocked: (locked: boolean) => void;
}

export const Sidebar: React.FC<SidebarProps> = ({ isLocked, setIsLocked }) => {
  const [isHovered, setIsHovered] = useState(false);
  const showFull = isLocked || isHovered;

  return (
    <aside
      className={`h-screen flex flex-col shrink-0 transition-all duration-300 ease-in-out border-r border-white/20 shadow-[20px_0_50px_rgba(0,0,0,0.2)] backdrop-blur-2xl ${showFull ? 'w-64' : 'w-16'} bg-transparent`}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
    >
      {/* Brand Section */}
      <div className={`h-16 flex items-center transition-all duration-300 border-b border-white/5 ${showFull ? 'px-4 gap-2.5' : 'justify-center'}`}>
        <div className={`flex-1 transition-all duration-300 overflow-hidden ${showFull ? 'opacity-100' : 'opacity-0 w-0'}`}>
          <span className="text-white flex gap-2 font-extrabold text-xl tracking-tight leading-none block">Boostup <br /> <span className="text-[#2fc6f6]">Innovix</span></span>
        </div>

        {showFull && (
          <button
            onClick={() => setIsLocked(!isLocked)}
            className="text-white/40 hover:text-white transition-colors p-1"
          >
            {isLocked ? <Pin size={14} className="text-[#2fc6f6]" /> : <PinOff size={14} />}
          </button>
        )}
      </div>

      <nav className="flex-1 overflow-y-auto overflow-x-hidden custom-scrollbar pt-4">
        <div className={`px-5 py-2 text-[10px] font-black text-white/30 uppercase tracking-[0.2em] transition-opacity duration-300 ${showFull ? 'opacity-100' : 'opacity-0 h-0 overflow-hidden'}`}>
          General
        </div>

        <SidebarItem icon={<LayoutDashboard size={18} />} label="Dashboard" to="/dashboard" showFull={showFull} />
        <SidebarItem icon={<Briefcase size={18} />} label="CRM" to="/crm" badge={2} showFull={showFull} />
        <SidebarItem icon={<LayoutGrid size={18} />} label="Collaboration" to="/collaboration" badge={3} showFull={showFull} />
        <SidebarItem icon={<CheckSquare size={18} />} label="Tasks" to="/tasks" badge={12} showFull={showFull} />
        <SidebarItem icon={<Users size={18} />} label="Employees" to="/employees" badge={7} showFull={showFull} />
        <SidebarItem icon={<MessageSquare size={18} />} label="Chats" to="/chats" badge={45} showFull={showFull} />

        <div className={`px-5 py-5 text-[10px] font-black text-white/30 uppercase tracking-[0.2em] transition-opacity duration-300 ${showFull ? 'opacity-100' : 'opacity-0 h-0 overflow-hidden'}`}>
          Sales Center
        </div>

        <SidebarItem icon={<Calendar size={18} />} label="Calendar" to="/calendar" showFull={showFull} />
        <SidebarItem icon={<Package size={18} />} label="Inventory" to="/inventory" showFull={showFull} />
        <SidebarItem icon={<Megaphone size={18} />} label="Marketing" to="/marketing" showFull={showFull} />
        <SidebarItem icon={<Globe size={18} />} label="Sites" to="/sites" showFull={showFull} />

        <div className={`px-5 py-5 text-[10px] font-black text-white/30 uppercase tracking-[0.2em] transition-opacity duration-300 ${showFull ? 'opacity-100' : 'opacity-0 h-0 overflow-hidden'}`}>
          Tools
        </div>

        <SidebarItem icon={<Bot size={18} />} label="Automation" to="/automation" showFull={showFull} />
        <SidebarItem icon={<Target size={18} />} label="Reports" to="/reports" showFull={showFull} />
        <SidebarItem icon={<Settings size={18} />} label="Settings" to="/settings" showFull={showFull} />
      </nav>
    </aside>
  );
};