import React, { useState } from 'react';
import { CRMBoard } from '../components/CRMBoard';
import { INITIAL_DEALS } from '../constants';
import type { Deal } from '../types';
import { ChevronDown, Layout, Pin } from 'lucide-react';

const KanbanPage: React.FC = () => {
    const [deals] = useState<Deal[]>(INITIAL_DEALS);
    const [activeTab, setActiveTab] = useState('Kanban');
    const [showUsers, setShowUsers] = useState(false);
    const [showPinnedDock, setShowPinnedDock] = useState(false);
    const [pinnedLinks, setPinnedLinks] = useState<{ id: string; title: string }[]>([
        { id: 'Kanban', title: 'Kanban' },
        { id: 'Inbound', title: 'Inbound' }
    ]);

    const togglePin = () => {
        if (pinnedLinks.some(l => l.id === activeTab)) {
            setPinnedLinks(pinnedLinks.filter(l => l.id !== activeTab));
        } else {
            setPinnedLinks([...pinnedLinks, { id: activeTab, title: activeTab }]);
        }
    };

    const isCurrentPinned = pinnedLinks.some(l => l.id === activeTab);

    return (
        <div className="flex-1 flex flex-col h-full overflow-hidden">
            {/* Minimal Tab Bar */}
            <div className="px-6 flex items-center justify-between h-14 shrink-0 bg-transparent border-b border-white/5">
                <div className="flex items-center gap-1 h-full">
                    {['Kanban', 'List', 'Activities', 'Calendar', 'Inbound', 'Planned'].map(tab => (
                        <button
                            key={tab}
                            onClick={() => setActiveTab(tab)}
                            className={`px-5 h-full transition-all flex items-center gap-2 font-bold text-[13px] border-b-2 font-[600]
                ${activeTab === tab ? 'text-white border-white' : 'text-white/70 border-transparent hover:text-white'}`}
                        >
                            {tab === 'Kanban' && <Layout size={16} className={activeTab === tab ? 'text-white' : 'text-white/60'} />}
                            {tab}
                            {tab === 'Inbound' && (
                                <span className="bg-red-500 text-white text-[10px] px-1.5 py-0.5 rounded-sm font-extrabold ml-1">
                                    2
                                </span>
                            )}
                        </button>
                    ))}
                </div>

                <button
                    onClick={togglePin}
                    className={`flex items-center gap-2 px-4 py-1.5 rounded-full transition-all duration-300 border ${isCurrentPinned
                        ? 'bg-amber-500/20 text-amber-400 border-amber-500/40 shadow-[0_0_15px_rgba(245,158,11,0.2)]'
                        : 'bg-white/5 text-white/40 border-white/10 hover:bg-white/10 hover:text-white'
                        }`}
                >
                    <Pin size={14} fill={isCurrentPinned ? "currentColor" : "none"} className={isCurrentPinned ? 'animate-pulse' : ''} />
                    <span className="text-[11px] font-bold tracking-tight">{isCurrentPinned ? 'PINNED' : 'PIN TO QUICK'}</span>
                </button>
            </div>

            {/* Board Workspace */}
            <div className="flex-1 flex overflow-hidden relative mt-2 px-1">
                <CRMBoard deals={deals} />

                {/* Bottom Right Actions Dock */}
                <div className="absolute right-8 bottom-8 flex items-end gap-5 z-50">
                    {/* Pinned Shortcuts Dock */}
                    <div className="flex flex-col items-center gap-3 relative group/pinned">
                        <div className={`absolute bottom-full mb-4 right-0 flex items-center gap-1 bg-slate-200/90 backdrop-blur-2xl p-1.5 rounded-full border border-white/30 shadow-[0_15px_35px_rgba(0,0,0,0.3)] transition-all duration-700 ease-[cubic-bezier(0.34,1.56,0.64,1)] origin-bottom-right ${showPinnedDock
                            ? 'opacity-100 translate-y-0 scale-100'
                            : 'opacity-0 translate-y-10 scale-50 pointer-events-none'
                            }`}>
                            <button
                                onClick={() => setShowPinnedDock(false)}
                                className="p-2 hover:bg-black/5 rounded-full transition-colors text-slate-500"
                            >
                                <ChevronDown size={20} className="rotate-90" />
                            </button>

                            <div className="flex items-center gap-1 px-1">
                                {pinnedLinks.map((link) => (
                                    <button
                                        key={link.id}
                                        onClick={() => setActiveTab(link.id)}
                                        className={`h-9 px-4 rounded-[10px] bg-white flex items-center justify-center shadow-sm border border-black/5 transition-all hover:shadow-md hover:-translate-y-0.5 ${activeTab === link.id ? 'ring-2 ring-blue-500/20' : ''
                                            }`}
                                    >
                                        <span className="text-[11px] font-bold text-slate-600 uppercase tracking-tight whitespace-nowrap">
                                            {link.title} {link.id === 'Inbound' ? 'MGM (...' : ''}
                                        </span>
                                    </button>
                                ))}
                            </div>

                            <div className="flex items-center gap-1 border-l border-black/5 ml-1 pl-1">
                                <button className="p-2 hover:bg-black/5 rounded-full transition-colors text-slate-400">
                                    <span className="flex gap-0.5">
                                        <div className="w-1 h-1 rounded-full bg-slate-400"></div>
                                        <div className="w-1 h-1 rounded-full bg-slate-400"></div>
                                        <div className="w-1 h-1 rounded-full bg-slate-400"></div>
                                    </span>
                                </button>
                            </div>
                        </div>

                        <button
                            onClick={() => setShowPinnedDock(!showPinnedDock)}
                            className={`w-14 h-14 rounded-full flex flex-col gap-1 items-center justify-center shadow-2xl border transition-all duration-500 ${showPinnedDock ? 'bg-slate-100 text-slate-600 border-white -translate-y-2' : 'bg-white text-slate-400 border-white/20 hover:scale-110 active:scale-95'
                                }`}
                        >
                            <div className={`w-6 h-0.5 bg-current rounded-full transition-all duration-500 ${showPinnedDock ? 'rotate-45 translate-y-1.5' : ''}`}></div>
                            <div className={`w-6 h-0.5 bg-current rounded-full transition-all duration-500 ${showPinnedDock ? 'opacity-0' : ''}`}></div>
                            <div className={`w-4 h-0.5 bg-current rounded-full self-center ml-2 transition-all duration-500 ${showPinnedDock ? '-rotate-45 -translate-y-1.5 w-6 ml-0' : ''}`}></div>
                        </button>
                    </div>

                    {/* Collaborative Users Section */}
                    <div className="flex flex-col items-center gap-3 group/dock">
                        <div className={`flex flex-col gap-2 bg-slate-200/50 backdrop-blur-[50px] p-2 rounded-full border border-white/20 shadow-[0_20px_50px_rgba(0,0,0,0.1)] transition-all duration-500 ease-[cubic-bezier(0.23,1,0.32,1)] origin-bottom ${showUsers
                            ? 'opacity-100 translate-y-0 scale-100'
                            : 'opacity-0 translate-y-12 scale-75 pointer-events-none'
                            }`}>
                            {[
                                { color: 'bg-pink-400', text: 'BF', count: 1 },
                                { color: 'bg-blue-400', text: 'JS' },
                                { color: 'bg-purple-400', text: 'AW' },
                                { color: 'bg-amber-400', text: 'TM' },
                                { color: 'bg-emerald-400', text: 'RK' }
                            ].map((user, i) => (
                                <div key={i} className="relative group/user">
                                    <div className={`w-10 h-10 rounded-full flex items-center justify-center text-[12px] font-extrabold text-white shadow-lg cursor-pointer hover:scale-110 transition-all border-2 border-white/20 active:scale-95 ${user.color}`}>
                                        {user.text}
                                    </div>
                                    {user.count && (
                                        <span className="absolute -top-1 -right-1 w-5 h-5 bg-red-500 text-white rounded-full text-[10px] flex items-center justify-center border-2 border-white font-black z-10">
                                            {user.count}
                                        </span>
                                    )}
                                </div>
                            ))}
                        </div>

                        <button
                            onClick={() => setShowUsers(!showUsers)}
                            className={`w-14 h-14 rounded-full flex items-center justify-center transition-all duration-500 shadow-2xl border backdrop-blur-3xl relative ${showUsers
                                ? 'bg-white text-red-500 border-white -translate-y-2'
                                : 'bg-white/10 text-white/50 border-white/10 hover:bg-white/15 hover:text-white active:scale-90'
                                }`}
                        >
                            <div className={`transition-all duration-500 ${showUsers ? 'rotate-180' : 'rotate-0'}`}>
                                <ChevronDown size={28} strokeWidth={2.5} />
                            </div>
                            {!showUsers && (
                                <span className="absolute -top-1.5 -right-1.5 w-6 h-6 bg-red-500 text-white rounded-full flex items-center justify-center font-black text-[11px] shadow-lg">
                                    5
                                </span>
                            )}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default KanbanPage;
