import React, { lazy } from 'react';
import { Routes, Route, Navigate, useLocation, Link } from 'react-router-dom';
import MainLayout from '../layouts/MainLayout';
import Drawer from '../components/common/Drawer';
import { Plus } from 'lucide-react';

// Lazy loaded pages
const DashboardPage = lazy(() => import('../pages/DashboardPage'));
const KanbanPage = lazy(() => import('../pages/KanbanPage'));

// Placeholder components for other routes
const PlaceholderPage = ({ title }: { title: string }) => (
    <div className="p-8">
        <h1 className="text-3xl font-bold mb-4">{title}</h1>
        <div className="bg-white/5 border border-white/10 p-12 rounded-3xl text-center">
            <p className="text-white/40 text-lg italic">This page is under construction. Stay tuned!</p>
        </div>
    </div>
);

const DealDetailContent = () => {
    const location = useLocation();
    return (
        <div className="space-y-6">
            <div className="p-4 bg-indigo-50 border border-indigo-100 rounded-xl mb-4">
                <div className="flex items-center justify-between mb-2">
                    <span className="text-[10px] font-black text-indigo-400 uppercase tracking-widest">Active Deal</span>
                    <span className="text-[10px] font-bold text-indigo-600 bg-white px-2 py-0.5 rounded-full shadow-sm border border-indigo-100 italic font-mono tracking-tighter">REF-2024-082</span>
                </div>
                <h2 className="text-2xl font-black text-slate-800 tracking-tight leading-none uppercase">Website Redesign Project</h2>
            </div>

            <Link
                to="/crm/add"
                state={{ backgroundLocation: location }}
                className="inline-block"
            >
                <button className="px-4 py-2 bg-slate-900 text-white rounded-xl flex items-center gap-2 hover:bg-indigo-600 transition-all font-bold text-sm shadow-lg shadow-indigo-100 uppercase tracking-wider">
                    <Plus size={16} strokeWidth={3} /> Add Related Deal
                </button>
            </Link>

            <div className="bg-white p-6 rounded-2xl border border-slate-100 shadow-sm transition-all hover:shadow-md">
                <h3 className="text-xs font-black text-slate-400 uppercase tracking-[0.2em] mb-4">Deal Information</h3>
                <div className="grid grid-cols-2 gap-6">
                    <div className="space-y-1">
                        <p className="text-[10px] text-slate-400 uppercase font-black">Client Name</p>
                        <p className="text-slate-700 font-bold text-lg leading-tight uppercase">Acme Corp Ltd.</p>
                    </div>
                    <div className="space-y-1 text-right">
                        <p className="text-[10px] text-slate-400 uppercase font-black">Deal Value</p>
                        <div className="flex items-baseline justify-end gap-1">
                            <span className="text-xs font-bold text-indigo-400">$</span>
                            <p className="text-indigo-600 font-black text-2xl leading-none">12,500</p>
                        </div>
                    </div>
                </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
                <div className="bg-emerald-50/50 p-4 rounded-2xl border border-emerald-100/50">
                    <p className="text-[10px] text-emerald-600 font-black uppercase mb-1">Status</p>
                    <p className="text-emerald-700 font-black text-sm uppercase">In Progress</p>
                </div>
                <div className="bg-amber-50/50 p-4 rounded-2xl border border-amber-100/50">
                    <p className="text-[10px] text-amber-600 font-black uppercase mb-1">Priority</p>
                    <p className="text-amber-700 font-black text-sm uppercase">Critical</p>
                </div>
            </div>

            <div className="bg-white p-6 rounded-2xl border border-slate-100 shadow-sm">
                <div className="flex items-center justify-between mb-6">
                    <h3 className="text-xs font-black text-slate-400 uppercase tracking-[0.2em]">Activity Timeline</h3>
                    <button className="text-[10px] font-black text-indigo-600 uppercase">View All</button>
                </div>
                <div className="space-y-6 relative before:absolute before:left-[7px] before:top-2 before:bottom-2 before:w-[2px] before:bg-slate-100">
                    {[
                        { title: 'Follow-up Call', time: '2:30 PM', status: 'Completed', color: 'bg-indigo-500' },
                        { title: 'Contract Drafted', time: 'Yesterday', status: 'Pending', color: 'bg-slate-300' },
                        { title: 'Initial Discovery', time: '2 days ago', status: 'Completed', color: 'bg-indigo-500' }
                    ].map((item, i) => (
                        <div key={i} className="flex gap-4 relative">
                            <div className={`w-4 h-4 rounded-full ${item.color} border-4 border-white shadow-sm shrink-0 z-10`} />
                            <div className="flex-1">
                                <div className="flex justify-between items-start">
                                    <p className="text-sm font-black text-slate-700 leading-tight uppercase">{item.title}</p>
                                    <span className="text-[10px] font-bold text-slate-400 italic">{item.time}</span>
                                </div>
                                <p className="text-xs text-slate-400 font-bold">{item.status}</p>
                            </div>
                        </div>
                    ))}
                </div>
            </div>
        </div>
    );
};

const AppRoutes: React.FC = () => {
    const location = useLocation();

    // backgroundLocation allows us to keep the context visible underneath
    const backgroundLocation = location.state?.backgroundLocation;

    return (
        <>
            {/* 1. Main Background Content (Always matches against the background location if available) */}
            <Routes location={backgroundLocation || location}>
                <Route path="/" element={<MainLayout />}>
                    <Route index element={<Navigate to="/dashboard" replace />} />
                    <Route path="dashboard" element={<DashboardPage />} />
                    <Route path="crm/*" element={<KanbanPage />} />
                    <Route path="collaboration" element={<PlaceholderPage title="Collaboration" />} />
                    <Route path="tasks" element={<PlaceholderPage title="Tasks and Projects" />} />
                    <Route path="employees" element={<PlaceholderPage title="Employees" />} />
                    <Route path="chats" element={<PlaceholderPage title="Chats" />} />
                    <Route path="calendar" element={<PlaceholderPage title="Calendar" />} />
                    <Route path="inventory" element={<PlaceholderPage title="Inventory" />} />
                    <Route path="marketing" element={<PlaceholderPage title="Marketing" />} />
                    <Route path="sites" element={<PlaceholderPage title="Sites" />} />
                    <Route path="automation" element={<PlaceholderPage title="Automation" />} />
                    <Route path="reports" element={<PlaceholderPage title="BI Analytics & Reports" />} />
                    <Route path="settings" element={<PlaceholderPage title="Settings" />} />
                    <Route path="*" element={<Navigate to="/dashboard" replace />} />
                </Route>
            </Routes>

            {/* 2. Modals/Drawers Layer */}
            {/* If we are at /crm/add and it has its own background state (like a deal details),
                we use that to keep the deal details visible. Otherwise we use the current location. */}
            <Routes>
                {/* Deal Detail Modal */}
                <Route
                    path="/crm/deal/:id"
                    element={
                        <Drawer title="Deal Details">
                            <DealDetailContent />
                        </Drawer>
                    }
                />

                {/* Add Deal Modal */}
                <Route
                    path="/crm/add"
                    element={
                        <Drawer title="Add New Deal">
                            <div className="p-4 flex flex-col items-center justify-center h-full text-center space-y-4">
                                <div className="w-16 h-16 bg-gradient-to-br from-indigo-500 to-purple-600 rounded-3xl flex items-center justify-center text-white shadow-2xl rotate-3">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="3" strokeLinecap="round" strokeLinejoin="round"><path d="M5 12h14"></path><path d="M12 5v14"></path></svg>
                                </div>
                                <div className="space-y-1">
                                    <h2 className="text-2xl font-black text-slate-800 uppercase tracking-tight">Create New Deal</h2>
                                    <p className="text-indigo-400 font-bold italic text-xs">Start your next success story.</p>
                                </div>
                                <div className="w-full space-y-5 pt-8">
                                    <div className="space-y-1.5 text-left">
                                        <label className="text-[10px] font-black text-slate-400 uppercase tracking-[0.2em] ml-1">Opportunity name</label>
                                        <input type="text" className="w-full bg-slate-50 border-2 border-slate-100 rounded-2xl px-5 py-4 text-slate-700 font-bold focus:outline-none focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500/30 transition-all placeholder:text-slate-300" placeholder="e.g. Website Overhaul" />
                                    </div>
                                    <div className="space-y-1.5 text-left">
                                        <label className="text-[10px] font-black text-slate-400 uppercase tracking-[0.2em] ml-1">Proposed budget</label>
                                        <div className="relative">
                                            <span className="absolute left-5 top-1/2 -translate-y-1/2 text-slate-300 font-black">$</span>
                                            <input type="number" className="w-full bg-slate-50 border-2 border-slate-100 rounded-2xl pl-10 pr-5 py-4 text-slate-700 font-bold focus:outline-none focus:ring-4 focus:ring-indigo-500/10 focus:border-indigo-500/30 transition-all placeholder:text-slate-300" placeholder="0.00" />
                                        </div>
                                    </div>
                                    <button className="w-full bg-indigo-600 hover:bg-slate-900 text-white font-black py-4 rounded-2xl shadow-xl shadow-indigo-100 transition-all transform hover:-translate-y-1 active:scale-[0.98] uppercase tracking-[0.1em] text-sm mt-4">Generate Deal</button>
                                </div>
                            </div>
                        </Drawer>
                    }
                />
            </Routes>

            {/* 3. Nested Modal logic: If we are in /crm/add AND we came from /crm/deal/:id,
                 we need to manually render the Deal Details underneath it because React Router Routes only matches ONE. */}
            {(location.pathname === '/crm/add' && backgroundLocation?.pathname?.match(/\/crm\/deal\/.+/)) && (
                <Routes location={backgroundLocation}>
                    <Route
                        path="/crm/deal/:id"
                        element={
                            <Drawer title="Deal Details">
                                <DealDetailContent />
                            </Drawer>
                        }
                    />
                </Routes>
            )}
        </>
    );
};

export default AppRoutes;
